<?php
/**
 * Elementor trait.
 *
 * @package Kitstarter
 * @since 1.0.0
 */

namespace Kitstarter\Traits;

use Elementor\Plugin;
use Elementor\Utils;
use Elementor\Controls_Stack;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

trait Elementor {

	/**
	 * Get template data.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function get_template_data() {
		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'kitstarter' ) ) {
			wp_send_json_error( esc_html__( 'Invalid nonce', 'kitstarter' ) );
		}

		if ( ! current_user_can( 'edit_posts' ) ) {
			wp_send_json_error( esc_html__( 'Permission denied', 'kitstarter' ) );
		}

		$id     = isset( $_POST['id'] ) ? sanitize_text_field( wp_unslash( $_POST['id'] ) ) : null;
		$entity = isset( $_POST['entity'] ) ? sanitize_text_field( wp_unslash( $_POST['entity'] ) ) : null;

		if ( ! $id || ! $entity ) {
			wp_send_json_error( esc_html__( 'Invalid request', 'kitstarter' ) );
		}

		$template_request = wp_remote_get(
			KITSTARTER_RESOURCE_URL . '/wp-json/kitstarter/v2/' . $entity . '/' . $id . '/components',
			array( 'sslverify' => false )
		);

		if ( is_wp_error( $template_request ) ) {
			wp_send_json_error( esc_html__( 'Failed to fetch template data', 'kitstarter' ) );
		}

		$template_data = wp_remote_retrieve_body( $template_request );
		$data          = json_decode( $template_data, true );

		if ( ! is_array( $data ) || empty( $data['success'] ) ) {
			wp_send_json_error( esc_html__( 'Failed to get template data', 'kitstarter' ) );
		}

		// Set the Request's state as an Elementor upload request, in order to support unfiltered file uploads.
		Plugin::$instance->uploads_manager->set_elementor_upload_state( true );

		if ( ! empty( $data['data']['content'] ) ) {
			$data['data']['content'] = $this->replace_elements_ids(
				$data['data']['content']
			);
			$data['data']['content'] = $this->process_export_import_content(
				$data['data']['content'],
				'on_import'
			);
		}

		// Save and transform global variables.
		if ( isset( $data['data']['metadata']['variables'] ) && ! empty( $data['data']['metadata']['variables']['data'] ) ) {
			$this->save_global_variables_to_kit( $data['data']['metadata']['variables']['data'] );
			$data['data']['metadata']['variables'] = $this->transform_global_variables(
				$data['data']['metadata']['variables']['data']
			);
		}

		// Save and transform global classes.
		if ( isset( $data['data']['metadata']['classes'] ) && ! empty( $data['data']['metadata']['classes']['items'] ) ) {
			$this->save_global_classes_to_kit( $data['data']['metadata']['classes'] );
			$data['data']['metadata']['classes'] = $this->transform_global_classes(
				$data['data']['metadata']['classes']
			);
		}

		// After the upload complete, set the elementor upload state back to false.
		Plugin::$instance->uploads_manager->set_elementor_upload_state( false );
		wp_send_json_success( $data['data'] );
	}

	/**
	 * Save global variables to the active kit.
	 *
	 * Merges new variables with existing ones in the kit, avoiding duplicates.
	 * Uses Elementor's format: { data: { id: variable }, watermark, version }.
	 *
	 * @since 1.0.0
	 * @param array $server_variables Variables from the server.
	 * @return void
	 */
	protected function save_global_variables_to_kit( $server_variables ) {
		$kit = Plugin::$instance->kits_manager->get_active_kit();

		if ( ! $kit ) {
			return;
		}

		$meta_key  = '_elementor_global_variables';
		$kit_data  = $kit->get_json_meta( $meta_key );
		$existing  = isset( $kit_data['data'] ) ? $kit_data['data'] : array();
		$watermark = isset( $kit_data['watermark'] ) ? $kit_data['watermark'] : 0;
		$version   = isset( $kit_data['version'] ) ? $kit_data['version'] : 1;

		// Merge new variables with existing, updating if ID already exists.
		foreach ( $server_variables as $var_id => $variable ) {
			// Skip deleted variables.
			if ( ! empty( $variable['deleted'] ) ) {
				continue;
			}

			// Check for duplicate labels (only for new variables, not updates).
			if ( ! isset( $existing[ $var_id ] ) ) {
				$label_exists = false;
				foreach ( $existing as $existing_id => $existing_var ) {
					if ( $existing_id === $var_id ) {
						continue;
					}
					if ( ! empty( $existing_var['deleted'] ) ) {
						continue;
					}
					if ( isset( $existing_var['label'] ) && strtolower( $existing_var['label'] ) === strtolower( $variable['label'] ) ) {
						$label_exists = true;
						break;
					}
				}

				if ( $label_exists ) {
					continue;
				}
			}

			// Transform value format.
			$value = $variable['value'];
			if ( is_array( $value ) && isset( $value['value'] ) ) {
				$value = $value['value'];
			}

			$existing[ $var_id ] = array(
				'type'  => $variable['type'],
				'label' => $variable['label'],
				'value' => $value,
				'order' => isset( $variable['order'] ) ? $variable['order'] : 1,
			);

			if ( ! empty( $variable['updated_at'] ) ) {
				$existing[ $var_id ]['updated_at'] = $variable['updated_at'];
			}
		}

		// Save back to kit.
		$kit->update_json_meta(
			$meta_key,
			array(
				'data'      => $existing,
				'watermark' => $watermark + 1,
				'version'   => $version,
			)
		);
	}

	/**
	 * Save global classes to the active kit.
	 *
	 * Merges new classes with existing ones in the kit, avoiding duplicates.
	 * Uses Elementor's format: { items: { id: class }, order: [] }.
	 *
	 * @since 1.0.0
	 * @param array $server_classes Classes from the server with items and order.
	 * @return void
	 */
	protected function save_global_classes_to_kit( $server_classes ) {
		$kit = Plugin::$instance->kits_manager->get_active_kit();

		if ( ! $kit ) {
			return;
		}

		$meta_key       = '_elementor_global_classes';
		$kit_data       = $kit->get_json_meta( $meta_key );
		$existing_items = isset( $kit_data['items'] ) ? $kit_data['items'] : array();
		$existing_order = isset( $kit_data['order'] ) ? $kit_data['order'] : array();

		$new_items = isset( $server_classes['items'] ) ? $server_classes['items'] : array();
		$new_order = isset( $server_classes['order'] ) ? $server_classes['order'] : array();

		// Merge new classes with existing, updating if ID already exists.
		foreach ( $new_order as $class_id ) {
			if ( ! isset( $new_items[ $class_id ] ) ) {
				continue;
			}

			$is_update = isset( $existing_items[ $class_id ] );

			// Check for duplicate labels (only for new classes, not updates).
			if ( ! $is_update ) {
				$label_exists = false;
				$new_label    = isset( $new_items[ $class_id ]['label'] ) ? $new_items[ $class_id ]['label'] : '';
				foreach ( $existing_items as $existing_id => $existing_class ) {
					if ( $existing_id === $class_id ) {
						continue;
					}
					if ( isset( $existing_class['label'] ) && strtolower( $existing_class['label'] ) === strtolower( $new_label ) ) {
						$label_exists = true;
						break;
					}
				}

				if ( $label_exists ) {
					continue;
				}
			}

			// Add or update class.
			$existing_items[ $class_id ] = $new_items[ $class_id ];

			// Add to order only if it's a new class.
			if ( ! $is_update ) {
				$existing_order[] = $class_id;
			}
		}

		// Save back to kit.
		$kit->update_json_meta(
			$meta_key,
			array(
				'items' => $existing_items,
				'order' => $existing_order,
			)
		);
	}

	/**
	 * Transform global variables from server format to localStorage format.
	 *
	 * Server format: { "e-gv-xxx": { type, label, value: { $$type, value }, order, ... } }
	 * localStorage format: { "e-gv-xxx": { type, label, value: "#color", order, ... } }
	 *
	 * @since 1.0.0
	 * @param array $server_variables Variables in server format.
	 * @return array Variables in localStorage format.
	 */
	protected function transform_global_variables( $server_variables ) {
		$result = array();

		foreach ( $server_variables as $id => $variable ) {
			// Skip deleted variables.
			if ( ! empty( $variable['deleted'] ) ) {
				continue;
			}

			// Extract value - server sends { $$type, value } but localStorage expects just the value string.
			$value = $variable['value'];
			if ( is_array( $value ) && isset( $value['value'] ) ) {
				$value = $value['value'];
			}

			$result[ $id ] = array(
				'type'  => $variable['type'],
				'label' => $variable['label'],
				'value' => $value,
				'order' => isset( $variable['order'] ) ? $variable['order'] : null,
			);

			if ( ! empty( $variable['updated_at'] ) ) {
				$result[ $id ]['updated_at'] = $variable['updated_at'];
			}
		}

		return $result;
	}

	/**
	 * Transform global classes from server format to localStorage format.
	 *
	 * Server format: { items: { "g-xxx": { id, type, label, variants } }, order: [] }
	 * localStorage format: [ { id, type, label, variants }, ... ]
	 *
	 * @since 1.0.0
	 * @param array $server_classes Classes in server format.
	 * @return array Classes in localStorage format (array of class objects).
	 */
	protected function transform_global_classes( $server_classes ) {
		$result = array();
		$items  = isset( $server_classes['items'] ) ? $server_classes['items'] : array();
		$order  = isset( $server_classes['order'] ) ? $server_classes['order'] : array();

		// Return classes in the order specified, as an array.
		foreach ( $order as $id ) {
			if ( isset( $items[ $id ] ) ) {
				$result[] = $items[ $id ];
			}
		}

		return $result;
	}

	/**
	 * Process export/import content.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param array  $content Any type of Elementor data.
	 * @param string $method  Method to call.
	 *
	 * @return mixed Iterated data.
	 */
	public function process_export_import_content( $content, $method ) {
		return Plugin::$instance->db->iterate_data(
			$content,
			function ( $element_data ) use ( $method ) {
				$element = Plugin::$instance->elements_manager->create_element_instance(
					$element_data
				);

				// If the widget/element doesn't exist, like a plugin that creates a widget but deactivated.
				if ( ! $element ) {
					return null;
				}

				return $this->process_element_export_import_content( $element, $method );
			}
		);
	}

	/**
	 * Replace elements IDs.
	 *
	 * For any given Elementor content/data, replace the IDs with new randomly
	 * generated IDs.
	 *
	 * @since 1.0.0
	 * @access protected
	 *
	 * @param array $content Any type of Elementor data.
	 *
	 * @return mixed Iterated data.
	 */
	protected function replace_elements_ids( $content ) {
		return Plugin::$instance->db->iterate_data(
			$content,
			function ( $element ) {
				$element['id'] = Utils::generate_random_string();

				return $element;
			}
		);
	}

	/**
	 * Process single element content for export/import.
	 *
	 * Process any given element and prepare the element data for export/import.
	 *
	 * @since 1.5.0
	 * @access protected
	 *
	 * @param Controls_Stack $element Element instance.
	 * @param string         $method  Method to call.
	 *
	 * @return array Processed element data.
	 */
	protected function process_element_export_import_content( Controls_Stack $element, $method ) {
		$element_data = $element->get_data();

		if ( method_exists( $element, $method ) ) {
			// TODO: Use the internal element data without parameters.
			$element_data = $element->{$method}( $element_data );
		}

		foreach ( $element->get_controls() as $control ) {
			$control_class = Plugin::$instance->controls_manager->get_control(
				$control['type']
			);

			// If the control doesn't exist, like a plugin that creates the control but deactivated.
			if ( ! $control_class ) {
				return $element_data;
			}

			if ( method_exists( $control_class, $method ) ) {
				$element_data['settings'][ $control['name'] ] = $control_class->{$method}(
					$element->get_settings( $control['name'] ),
					$control
				);
			}

			// On Export, check if the control has an argument 'export' => false.
			if (
				'on_export' === $method &&
				isset( $control['export'] ) &&
				false === $control['export']
			) {
				unset( $element_data['settings'][ $control['name'] ] );
			}
		}

		return $element_data;
	}
}
