<?php
/**
 * Main Kitstarter class
 *
 * @package Kitstarter
 * @since   1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Kitstarter
 *
 * Main plugin class (placeholder for future functionality).
 *
 * @since 1.0.0
 */
class Kitstarter {

	/**
	 * Check if Elementor V4 features are active
	 *
	 * @return bool
	 */
	public function is_v4_compatible() {
		if ( ! class_exists( '\Elementor\Plugin' ) ) {
			return false;
		}

		$experiments = \Elementor\Plugin::$instance->experiments;

		return $experiments->is_feature_active( 'e_variables' ) && $experiments->is_feature_active( 'e_classes' );
	}

	/**
	 * Add menu to wp admin
	 *
	 * @return void
	 */
	public function admin_menu() {
		add_menu_page(
			__( 'Kitstarter', 'kitstarter' ),
			'Kitstarter',
			'manage_options',
			'kitstarter',
			function () {},
			KITSTARTER_URL . 'public/images/kitstarter-logo.svg',
			51
		);

		// Register settings page without adding to menu (null parent).
		// The menu link is added via $submenu array in admin_menu_addition().
		if ( $this->is_v4_compatible() ) {
			add_submenu_page(
				null, // Hidden from menu.
				__( 'Settings', 'kitstarter' ),
				__( 'Settings', 'kitstarter' ),
				'manage_options',
				'kitstarter-settings',
				array( $this, 'render_settings_page' )
			);
		}
	}

	/**
	 * Render settings page
	 *
	 * @return void
	 */
	public function render_settings_page() {
		$editor_version = get_option( 'kitstarter_editor_version', 'v3' );
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Kitstarter Settings', 'kitstarter' ); ?></h1>
			<form id="kitstarter-settings-form" method="post">
				<?php wp_nonce_field( 'kitstarter_settings', 'kitstarter_settings_nonce' ); ?>
				<table class="form-table" role="presentation">
					<tbody>
						<tr>
							<th scope="row">
								<label for="kitstarter_editor_version"><?php esc_html_e( 'Editor Version', 'kitstarter' ); ?></label>
							</th>
							<td>
								<select name="kitstarter_editor_version" id="kitstarter_editor_version">
									<option value="v3" <?php selected( $editor_version, 'v3' ); ?>><?php esc_html_e( 'Standard (Elementor 3.x)', 'kitstarter' ); ?></option>
									<option value="v4" <?php selected( $editor_version, 'v4' ); ?>><?php esc_html_e( 'V4 (Elementor 4.x)', 'kitstarter' ); ?></option>
								</select>
								<p class="description"><?php esc_html_e( 'Select which component library version to use. V4 includes support for global variables and classes.', 'kitstarter' ); ?></p>
							</td>
						</tr>
					</tbody>
				</table>
				<?php submit_button( __( 'Save Settings', 'kitstarter' ) ); ?>
			</form>
			<div id="kitstarter-settings-message" style="display: none;"></div>
		</div>
		<?php
	}

	/**
	 * Handle AJAX save settings
	 *
	 * @return void
	 */
	public function ajax_save_settings() {
		// Verify nonce.
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'kitstarter_settings' ) ) {
			wp_send_json_error( array( 'message' => __( 'Security check failed.', 'kitstarter' ) ) );
		}

		// Check capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'You do not have permission to do this.', 'kitstarter' ) ) );
		}

		// Get and validate editor version.
		$editor_version = isset( $_POST['editor_version'] ) ? sanitize_text_field( wp_unslash( $_POST['editor_version'] ) ) : 'v3';
		if ( ! in_array( $editor_version, array( 'v3', 'v4' ), true ) ) {
			$editor_version = 'v3';
		}

		// Save option.
		update_option( 'kitstarter_editor_version', $editor_version );

		wp_send_json_success( array( 'message' => __( 'Settings saved successfully.', 'kitstarter' ) ) );
	}

	/**
	 * Add menu to wp admin
	 *
	 * @return void
	 */
	public function admin_menu_addition() {
		global $submenu;

		// Remove the auto-generated first submenu item (duplicate of parent).
		// Only remove if slug is 'kitstarter' AND title is 'Kitstarter' (the auto-generated one).
		// This preserves other submenus like 'License' that may use the same slug.
		if ( isset( $submenu['kitstarter'] ) ) {
			foreach ( $submenu['kitstarter'] as $key => $item ) {
				if ( isset( $item[2] ) && 'kitstarter' === $item[2] && isset( $item[0] ) && 'Kitstarter' === $item[0] ) {
					// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Removing auto-generated duplicate submenu.
					unset( $submenu['kitstarter'][ $key ] );
					break;
				}
			}
		}

		// Add Settings submenu only if V4 features are active.
		if ( $this->is_v4_compatible() ) {
			// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Adding custom submenu item.
			$submenu['kitstarter'][] = [
				__( 'Settings', 'kitstarter' ),
				'manage_options',
				'admin.php?page=kitstarter-settings',
				__( 'Settings', 'kitstarter' ),
			];
		}

		if ( ! kitstarter_has_pro() || ! kitstarter_license_valid() ) {
			// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Adding custom submenu item.
			$submenu['kitstarter'][] = [
				'Get Pro',
				'manage_options',
				'https://kitstarter.io/pricing/',
				'Get Pro',
			];
		}

		// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- Adding custom submenu item.
		$submenu['kitstarter'][] = [
			'Get help',
			'manage_options',
			'http://kitstarter.io/',
			'Get help',
		];
	}

	/**
	 * Enqueue admin scripts
	 *
	 * @param string $hook The current admin page hook suffix.
	 * @return void
	 */
	public function enqueue_admin_scripts( $hook ) {
		wp_enqueue_style( 'kitstarter-admin', KITSTARTER_URL . 'public/css/admin.css', array(), KITSTARTER_VERSION );

		// Enqueue settings page script.
		if ( 'admin_page_kitstarter-settings' === $hook ) {
			wp_enqueue_script( 'kitstarter-settings', KITSTARTER_URL . 'public/js/settings.js', array( 'jquery' ), KITSTARTER_VERSION, true );
			wp_localize_script(
				'kitstarter-settings',
				'kitstarter_settings',
				array(
					'ajax_url' => admin_url( 'admin-ajax.php' ),
					'nonce'    => wp_create_nonce( 'kitstarter_settings' ),
				)
			);
		}
	}

	/**
	 * Add action links to the plugin page.
	 *
	 * @param array $links Plugin action links.
	 * @return array Modified plugin action links.
	 */
	public function add_action_links( $links ) {
		if ( ! kitstarter_has_pro() || ! kitstarter_license_valid() ) {
			$links[] =
			'<a href="https://kitstarter.io/pricing/" target="_blank" style="font-weight: 700; color: #268D61;">Get Kitstarter Pro</a>';
		}

		return $links;
	}

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_filter( 'plugin_action_links_' . KITSTARTER_BASENAME, array( $this, 'add_action_links' ) );
		add_action( 'admin_menu', array( $this, 'admin_menu' ), 5 );
		add_action( 'admin_menu', array( $this, 'admin_menu_addition' ), 100 );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
		add_action( 'wp_ajax_kitstarter_save_settings', array( $this, 'ajax_save_settings' ) );
	}
}
