<?php
/**
 * Builder class for Kitstarter
 *
 * Responsible for injecting the component library UI into Elementor editor.
 *
 * @package Kitstarter
 * @since   1.0.0
 */

namespace Kitstarter;

use Kitstarter\Traits\Elementor;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Builder
 *
 * Handles the Elementor editor integration.
 *
 * @since 1.0.0
 */
class Builder {

	use Elementor;

	/**
	 * Generate app wrapper div.
	 *
	 * Outputs the container element for the React application.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function wrapper() {
		echo '<div id="kitstarter-builder"></div>';
	}

	/**
	 * Enqueue preview scripts and styles.
	 *
	 * Loads the CSS for the Elementor preview iframe.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function enqueue_preview_scripts() {
		$script_data = require KITSTARTER_PATH . 'build/builder.asset.php';
		wp_enqueue_style(
			'kitstarter-builder',
			KITSTARTER_URL . 'public/css/builder.css',
			array(),
			$script_data['version']
		);
	}

	/**
	 * Enqueue editor scripts.
	 *
	 * Loads the JavaScript for the Elementor editor.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function enqueue_editor_scripts() {
		$script_data = require KITSTARTER_PATH . 'build/builder.asset.php';
		wp_enqueue_script(
			'kitstarter-builder',
			KITSTARTER_URL . 'build/builder.js',
			$script_data['dependencies'],
			$script_data['version'],
			true
		);
		wp_localize_script( 'kitstarter-builder', 'kitstarter', apply_filters( 'kitstarter/data/scripts', $this->get_localize_data() ) );
		wp_enqueue_style(
			'kitstarter-builder',
			KITSTARTER_URL . 'build/style.css',
			array(),
			$script_data['version']
		);
	}

	/**
	 * Check if Elementor V4 features are active
	 *
	 * @return bool
	 */
	public function is_v4_compatible() {
		if ( ! class_exists( '\Elementor\Plugin' ) ) {
			return false;
		}

		$experiments = \Elementor\Plugin::$instance->experiments;

		return $experiments->is_feature_active( 'e_variables' ) && $experiments->is_feature_active( 'e_classes' );
	}

	/**
	 * Get localize data
     *
	 * @return array
	 */
	public function get_localize_data() {
		return [
			'version'       => KITSTARTER_VERSION,
			'authToken'     => base64_encode( home_url() ), // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
			'savedElements' => get_option( 'kitstarter_saved_elements', [] ),
			'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
			'ajaxNonce'     => wp_create_nonce( 'kitstarter' ),
			'editorVersion' => $this->is_v4_compatible() ? get_option( 'kitstarter_editor_version', 'v3' ) : 'v3',
		];
	}

	/**
	 * Save element
     *
	 * @return void
	 */
	public function save_template() {
		// Check nonce.
		$nonce = isset( $_POST['_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['_nonce'] ) ) : '';
		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'kitstarter' ) ) {
			wp_send_json_error();
		}

		// Check if template_id is set.
		if ( ! isset( $_POST['template_id'] ) || ! is_string( $_POST['template_id'] ) ) {
			wp_send_json_error( 'Invalid component ID' );
		}

		// Elements is array of element_ids.
		$elements    = get_option( 'kitstarter_saved_elements', [] );
		$template_id = sanitize_text_field( wp_unslash( $_POST['template_id'] ) );

		// Add or remove element_id from elements.
		if ( in_array( $template_id, $elements, true ) ) {
			$elements = array_diff( $elements, [ $template_id ] );
			$elements = array_values( $elements );
		} else {
			$elements[] = $template_id;
		}

		update_option( 'kitstarter_saved_elements', $elements );

		wp_send_json_success( $elements );
	}

	/**
	 * Constructor.
	 *
	 * Registers hooks for Elementor integration.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'elementor/editor/footer', array( $this, 'wrapper' ) );
		add_action( 'elementor/editor/after_enqueue_scripts', array( $this, 'enqueue_editor_scripts' ) );
		add_action( 'elementor/preview/enqueue_scripts', array( $this, 'enqueue_preview_scripts' ) );
		add_action( 'wp_ajax_kitstarter_get_template_data', array( $this, 'get_template_data' ) );
		add_action( 'wp_ajax_kitstarter_save_template', array( $this, 'save_template' ) );
	}
}
