<?php
/**
 * Autoloader class for Kitstarter
 *
 * Responsible for loading all class files.
 *
 * @package Kitstarter
 * @since   1.0.0
 */

namespace Kitstarter;

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Autoloader
 *
 * Handles autoloading of plugin classes.
 *
 * @since 1.0.0
 */
class Autoloader {

	/**
	 * Classes map.
	 *
	 * @since 1.0.0
	 * @var array
	 */
	private static $classes_map;

	/**
	 * Initialize the class map.
	 *
	 * @since 1.0.0
	 * @return array Class name to file path mapping.
	 */
	private static function init_class_map() {
		return array(
			'Kitstarter\\Builder'           => KITSTARTER_PATH . 'app/class-builder.php',
			'Kitstarter\\Traits\\Elementor' => KITSTARTER_PATH . 'app/traits/trait-elementor.php',
			'Kitstarter'                    => KITSTARTER_PATH . 'app/class-kitstarter.php',
		);
	}

	/**
	 * Get the class map.
	 *
	 * @since 1.0.0
	 * @return array Class name to file path mapping.
	 */
	private static function get_class_map() {
		if ( ! self::$classes_map ) {
			self::$classes_map = self::init_class_map();
		}

		return self::$classes_map;
	}

	/**
	 * Run the autoloader.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public static function run() {
		spl_autoload_register( array( __CLASS__, 'autoload' ) );

		self::load_classes();
	}

	/**
	 * Autoload callback.
	 *
	 * @since 1.0.0
	 * @param string $class_name Class name to autoload.
	 * @return void
	 */
	private static function autoload( $class_name ) {
		$class_map = self::get_class_map();

		if ( isset( $class_map[ $class_name ] ) ) {
			require $class_map[ $class_name ];
		}
	}

	/**
	 * Load all registered classes.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	private static function load_classes() {
		$class_map = self::get_class_map();

		foreach ( $class_map as $class_name => $path ) {
			if ( trait_exists( $class_name ) ) {
				continue;
			}
			new $class_name();
		}
	}
}
